﻿using System;
using System.Diagnostics;
using System.Net;
using System.Net.Sockets;
using System.Reflection;
using FyndSharp.Utilities.Common;
using gov.va.med.vbecs.Common.Log;
using gov.va.med.VBECS.Communication.Common;

namespace gov.va.med.VBECS.Communication.Channels
{
#if NUNIT
    public
#else
    internal 
#endif  
    class TcpChannel : BaseChannel
    {
        private const int RECEIVING_BUFFER_SIZE = 4096; //4KB
        private readonly Socket _clientSocket;
        private readonly Object _lockObject;
        private readonly byte[] _receivingBuffer;

        private volatile bool _isRunning;
        // Logger object
        readonly ILogger _logger = LogManager.Instance().LoggerLocator.GetLogger(MethodBase.GetCurrentMethod().DeclaringType);

        public TcpChannel(Socket theClientSocket)
            : base(theClientSocket == null ? null : (IPEndPoint) theClientSocket.RemoteEndPoint)
        {
            Checker.NotNull(theClientSocket);

            _receivingBuffer = new byte[RECEIVING_BUFFER_SIZE];
            _lockObject = new Object();
            _clientSocket = theClientSocket;
        }

        protected override void SendImpl(IMessage aMessage)
        {
            //Send message
            var totalSent = 0;
            lock (_lockObject)
            {
                //Create a byte array from message according to current protocol
                var messageBytes = Protocol.GetBytes(aMessage);
                //Send all bytes to the remote application
                while (totalSent < messageBytes.Length)
                {
                    var sent = _clientSocket.Send(messageBytes, totalSent, messageBytes.Length - totalSent,
                                                  SocketFlags.None);
                    if (sent <= 0)
                    {
                        throw new CommunicationException("Message could not be sent via TCP socket. Only " + totalSent +
                                                         " bytes of " + messageBytes.Length + " bytes are sent.");
                    }

                    totalSent += sent;
                }
            }
        }

        protected override void StartImpl()
        {
            _isRunning = true;
            _clientSocket.BeginReceive(_receivingBuffer
                                       , 0
                                       , _receivingBuffer.Length
                                       , 0
                                       , receive_callback
                                       , null);
        }

        protected override void DisconnectImpl()
        {
            if (Status != CommunicationStatus.Connected)
            {
                return;
            }

            _isRunning = false;
            try
            {
                _clientSocket.Close();
            }
#if TRACE
            catch (Exception e)
            {
                Trace.WriteLine(e.ToString());
            }
#else
            catch { }
#endif
        }


        private void receive_callback(IAsyncResult ar)
        {
            if (!_isRunning)
            {
                return;
            }

            try
            {
                //Get received bytes count
                var bytesRead = _clientSocket.EndReceive(ar);
                if (bytesRead > 0)
                {
                    LastReceivedTime = DateTime.Now;

                    //Copy received bytes to a new byte array
                    var receivedBytes = new byte[bytesRead];
                    Array.Copy(_receivingBuffer, 0, receivedBytes, 0, bytesRead);

                    //Read messages according to current wire protocol
                    var messages = Protocol.BuildMessages(receivedBytes);

                    //Raise MessageReceived event for all received messages
                    foreach (var message in messages)
                    {
                        FireMessageReceivedEvent(message);
                        if (message is ICloseConnectionResponceMessage)
                            _isRunning = false;
                    }
                }
                else
                {
                    throw new CommunicationException("Tcp socket is closed");
                }

                //Read more bytes if still running
                if (_isRunning)
                {
                    _clientSocket.BeginReceive(_receivingBuffer
                        , 0
                        , _receivingBuffer.Length
                        , 0
                        , receive_callback
                        , null);
                }
            }
#if TRACE
            catch (Exception e)
            {
                Disconnect();
                Trace.WriteLine(e.ToString());
            }
#else
            catch 
            {
                this.Disconnect();
            }

#endif
        }
    }
}